      subroutine WHAM_CALC(remd_t_bath)
! Weighed Histogram Analysis Method (WHAM) code
! Written by A. Liwo based on the work of Kumar et al., 
! J.Comput.Chem., 13, 1011 (1992)
!
! 2/1/05 Multiple temperatures allowed.
! 2/2/05 Free energies calculated directly from data points
!  acc. to Eq. (21) of Kumar et al.; final histograms also
!  constructed based on this equation.
!
! Adapted to replica-average restrained sampling (temperature replicas only)
! May 19, 2025
      implicit none
      include "DIMENSIONS"
#ifdef MPI
      include "mpif.h"
      include "COMMON.SETUP"
      integer ierror,errcode,status(MPI_STATUS_SIZE) 
#endif
      include "COMMON.CONTROL"
      include "COMMON.IOUNITS"
      include "COMMON.MD"
      include "COMMON.REMD"
      include "COMMON.NMR"
      include "COMMON.WHAM"
      include "COMMON.FFIELD"
      double precision finorm_max,potfac,vf
      parameter (finorm_max=1.0d0)
      integer i,ii,j,jj,k,kk,ib,l,m,ind,iter,t
      double precision v(max_cg_procs,max_rep),
     & entfac(max_cg_procs)
      double precision remd_t_bath(max_cg_procs)
      double precision energia(0:n_ene),enetb(0:n_ene,max_cg_procs)
#ifdef MPI
      logical lprint /.true./
#endif
      double precision betaT,weight,econstr
      double precision fi(max_rep),fimax(max_rep),f(max_rep),
     & denom,finorm,fimin /1.0d-3/,avefi,pom,potEmin,ent,vmax,aux
      double precision etot
 
#ifdef DEBUG
      write(iout,*) "Energies of my conofrmation"
      call enerprint(potEcomp(0))
#endif
      call MPI_Gather(potEcomp(0),n_ene+1,MPI_DOUBLE_PRECISION,
     &             enetb(0,1),n_ene+1,MPI_DOUBLE_PRECISION,king,
     &             CG_COMM,IERROR)

      if (me.ne.king) return
#ifdef DEBUG
      write (iout,*) "enetb array"
      do i=0,n_ene
        write (iout,'(i5,30f10.5)') i,(enetb(i,j),j=1,nodes)
      enddo
#endif
#ifdef WHAMOUT
      write (iout,'(a)') "Solving WHAM equations"
#endif
      potEmin=enetb(0,1)
      do i=2,nodes
        if (enetb(0,i).le.potEmin) potEmin=enetb(0,i)
      enddo ! i
c#define DEBUG
#ifdef DEBUG
      write (iout,*) "remd_t",remd_t(:nrep)
      write (iout,*) "beta_h",beta_h(:nrep)
#endif
! Compute the Boltzmann factor corresponing to restrain potentials in different
! simulations.
      do i=1,nodes
#ifdef DEBUG
        write (iout,'(i5,31f8.2)') i,(enetb(k,i),k=1,n_ene)
#endif
c#ifdef DEBUG
c        write (iout,*) wsc,wscp,welec,wvdwpp,wang,wtor,wscloc,
c     &    wcorr,wcorr5,wcorr6,wturn4,wturn3,wturn6,wel_loc,
c     &    wtor_d,wsccor,wbond
c#endif
        do ib=1,nrep
          call rescale_weights(remd_t(ib))
          call sum_energy(enetb(0,i),.false.)
          etot=enetb(0,i)
#ifdef DEBUG
          write (iout,*) i,ib,1.0d0/(beta_h(ib)*1.987D-3),etot,potEmin
#endif
#ifdef DEBUG
          write (iout,*)"Conformation",i
          call enerprint(enetb(0,i))
#endif
          v(i,ib)=-beta_h(ib)*(etot-potEmin)
#ifdef DEBUG
          write (iout,'(2i5,f10.3,4e15.5)') i,ib,
     &           1.0d0/(beta_h(ib)*1.987D-3),
     &           etot,potEmin,etot-potEmin,v(i,ib)
#endif
        enddo ! ib
      enddo     ! i
#ifdef DEBUG
      write (iout,*) "The V array"
      do t=1,nodes
        write (iout,'(i5,80f10.2)') t,(v(t,ib),ib=1,nrep)
      enddo
#endif
! Simple iteration to calculate free energies corresponding to all simulation
! runs.
      f(:nrep)=0.0d0
      do iter=1,maxit_wham
        
! Compute new free-energy values corresponding to the righ-hand side of the 
! equation and their derivatives.
#ifdef WHAMOUT
        write (iout,*) "------------------------fi"
#endif
        do t=1,nodes
          vmax=-1.0d+20
          do k=1,nrep
            vf=v(t,k)+f(k)
            if (vf.gt.vmax) vmax=vf
          enddo
          denom=0.0d0
          do k=1,nrep
            aux=f(k)+v(t,k)-vmax
#ifdef DEBUG
            write (iout,*) "t",t," k",k," aux",aux
#endif
            if (aux.gt.-200.0d0)
     &        denom=denom+remd_m(k)*dexp(aux)
          enddo
          entfac(t)=-dlog(denom)-vmax
#ifdef DEBUG
          write (iout,*) t,"vmax",vmax," denom",denom,"entfac",entfac(t)
#endif
        enddo

        do ib=1,nrep
          fimax(ib)=v(1,ib)+entfac(1)
          do t=2,nodes
            if (v(t,ib)+entfac(t).gt.fimax(ib))
     &        fimax(ib)=v(t,ib)+entfac(t)
          enddo
#ifdef DEBUG
          write (iout,*) "ib",ib," fimax",fimax(ib)
#endif
        enddo ! ib
        do ib=1,nrep
          fi(ib)=0.0d0
          do t=1,nodes
            fi(ib)=fi(ib)+dexp(v(t,ib)+entfac(t)-fimax(ib))
#ifdef DEBUG
            write (iout,*) "t",t," ib",ib," v+entfac",v(t,ib)+entfac(t),
     &        "fimax",fimax(ib)
            write (iout,'(2i5,3e15.5)') t,ib,v(t,ib),entfac(t),fi(ib)
#endif
          enddo
        enddo ! ib

#ifdef DEBUG
        do ib=1,nrep
          write (iout,*) " ib",ib," beta=",beta_h(ib)," fi",fi(ib)
        enddo
#endif
        avefi=0.0d0
        do ib=1,nrep
          fi(ib)=-dlog(fi(ib))-fimax(ib)
          avefi=avefi+fi(ib)
        enddo
        avefi=avefi/nrep
        do ib =1,nrep
          fi(ib)=fi(ib)-avefi
#ifdef WHAMOUT
          write (iout,*) "beta=",beta_h(ib)," fi",fi(ib),f(ib)
#endif
        enddo

! Compute the norm of free-energy increments.
        finorm=0.0d0
        do ib=1,nrep
          finorm=finorm+dabs(fi(ib)-f(ib))
          f(ib)=fi(ib)
        enddo

#ifdef WHAMOUT
        write (iout,*) 'Iteration',iter,' finorm',finorm
#endif
! Exit, if the increment norm is smaller than pre-assigned tolerance.
        if (finorm.lt.fimin) then
#ifdef WHAMOUT
          write (iout,*) 'Iteration converged'
#endif
          goto 20
        endif

      enddo ! iter

      write (iout,*) "Warning: WHAM did not converge"

   20 continue

#ifdef DEBUG
      write (iout,*) "entfac",(entfac(i),i=1,nodes)
#endif
c
c Calculate weight table
c
      do i=1,nodes
        call rescale_weights(remd_t_bath(i))
        do ii=1,nodes
          aux=1.0d0/(1.987d-3*remd_t_bath(i))
          call sum_energy(enetb(0,ii),.false.)
          rep_weight(ii,i)=dexp(entfac(ii)-aux*(enetb(0,ii)-potEmin))
#ifdef DEBUG
          write(iout,*) "i",i," ii",ii," remd_t_bath(i)",remd_t_bath(i),
     &      " aux",aux," energy",enetb(0,ii)," sum",
     &      entfac(ii)-aux*(enetb(0,ii)-potEmin),
     &      " weight", rep_weight(ii,i)
#endif
        enddo
      enddo
      do i=1,nodes
        aux=0.0d0
        do ii=1,nodes
          aux=aux+rep_weight(ii,i)
        enddo
        rep_weight(:nodes,i)=rep_weight(:nodes,i)/aux
      enddo

      call rescale_weights(t_bath)

      return
c#undef DEBUG
      end
