
c                             DRIVER 1
c     --------------------------------------------------------------
c                SIMPLE DRIVER FOR L-BFGS-B (version 2.3)
c     --------------------------------------------------------------
c
c        L-BFGS-B is a code for solving large nonlinear optimization
c        problems with simple bounds on the variables.
c
c        The code can also be used for unconstrained problems and is
c        as efficient for these problems as the earlier limited memory
c        code L-BFGS.
c
c        This is the simplest driver in the package. It uses all the
c        default settings of the code.
c
c
c     References:
c
c        [1] R. H. Byrd, P. Lu, J. Nocedal and C. Zhu, ``A limited
c        memory algorithm for bound constrained optimization'',
c        SIAM J. Scientific Computing 16 (1995), no. 5, pp. 1190--1208.
c
c        [2] C. Zhu, R.H. Byrd, P. Lu, J. Nocedal, ``L-BFGS-B: FORTRAN
c        Subroutines for Large Scale Bound Constrained Optimization''
c        Tech. Report, NAM-11, EECS Department, Northwestern University,
c        1994.
c
c        (Postscript files of these papers are available via anonymous
c        ftp to ece.nwu.edu in the directory pub/lbfgs/lbfgs_bcm.)
c
c                              *  *  *
c
c        NEOS, November 1994. (Latest revision April 1997.)
c        Optimization Technology Center.
c        Argonne National Laboratory and Northwestern University.
c        Written by
c                           Ciyou Zhu
c        in collaboration with R.H. Byrd, P. Lu-Chen and J. Nocedal.
c
c     NOTE: The user should adapt the subroutine 'timer' if 'etime' is
c           not available on the system.  An example for system 
c           AIX Version 3.2 is available at the end of this driver.
c
c     **************

      subroutine lbfgsb_driver(n,ib,fit,shannon)
c
c 12/28/2023 AL compute conformation-averaged NOEs
c
      implicit none
      include 'DIMENSIONS'
      include 'sizesclu.dat'
c     This simple driver demonstrates how to call the L-BFGS-B code to
c     solve a sample problem (the extended Rosenbrock function 
c     subject to bounds on the variables). The dimension n of this
c     problem is variable.
 
      integer          nmax, mmax, lenwa
      parameter       (nmax  = maxconf, mmax = 17)
      parameter       (lenwa = 2*mmax*nmax +  4*nmax
     +                      + 11*mmax*mmax + 8*mmax)

c     nmax  is the dimension of the largest problem to be solved.
c     mmax  is the maximum number of limited memory corrections.
c     lenwa is the corresponding real workspace required.
#ifdef MPI
      include "mpif.h"
      integer IERR
      include "COMMON.MPI"
#endif
      include 'COMMON.CONTROL'
      include 'COMMON.CLUSTER'
      include 'COMMON.CHAIN'
      include 'COMMON.INTERACT'
      include 'COMMON.VAR'
      include 'COMMON.TEMPFAC'
      include 'COMMON.IOUNITS'
      include 'COMMON.SBRIDGE'
      include 'COMMON.NMR'
      integer nviol
      common /nmrviol/ nviol
      integer ib,imin,ii,j,jj,iti,itj,k,jcon,itot,ip
      double precision qpart,shannon,fit,totfreemin
      double precision small /1.0d-15/ 
 
c     Declare the variables needed by the code.
c     A description of all these variables is given at the end of 
c     the driver.
 
      character*60     task, csave
      logical          lsave(4)
      integer          n, m, iprint,
     +                 nbd(nmax), iwa(3*nmax), isave(44)
      double precision f, factr, pgtol, 
     +                 x(nmax), l(nmax), u(nmax), g(nmax), dsave(29), 
     +                 wa(lenwa)
      double precision x0(nmax),xnorm(nmax)
#ifdef CHECKGRAD
      double precision g_(nmax),aux,xi
#endif

c     Declare a few additional variables for this sample problem.

      double precision t1, t2
      integer          i
 
      write (iout,*) "Calling lbfgs_b"
      call flush(iout)
c     We wish to have output at every iteration.

      iprint = 1

c     We specify the tolerances in the stopping criteria.

      factr  = 1.0d+7
      pgtol  = 1.0d-5

c     We specify the dimension n of the sample problem and the number
c     m of limited memory corrections stored.  (n and m should not
c     exceed the limits nmax and mmax respectively.)
 
c      n      = 25
c      n      = 5000
      m      =  5
 
c     We now provide nbd which defines the bounds on the variables:
c     l   specifies the lower bounds,
c     u   specifies the upper bounds. 
 
      do 10 i = 1, n
         nbd(i) = 1
c         l(i)   = 1.0d-20
c         u(i)   = 1.0d0
         l(i)   = -20.0d0
         u(i)   = 0.0d0
   10 continue
      if (canon) then
        x0(:n)=1.0d0/(n+1.0d0)
        totfreemin=0.0d0
        imin=1
      else   
c        call enecalc(ib,n)
        totfreemin=totfree(1)
        imin=1
        do i=2,n
          if (totfree(i).lt.totfreemin) then
            totfreemin=totfree(i)
            imin=i
          endif
        enddo
c        write (iout,*) "totfreemin",totfreemin
        qpart=0.0d0
        do i=1,n
c          write (iout,*) "i",i," totfree",totfree(i),
c     &      totfree(i)-totfreemin
          x0(i)=dexp(-totfree(i)+totfreemin)
          if (x0(i).lt.small) x0(i)=small
c          write (iout,*) "x0",x0(i)
          qpart=qpart+x0(i) 
        enddo
c        write (iout,*) "qpart",qpart
        do i=1,n
          x0(i)=x0(i)/qpart
c          write (iout,*) "x0",i,x0(i)
        enddo
      endif
c     First set bounds on the odd-numbered variables.

      do 14 i = 1, n
c         x(i) = x0(i)
c         x(i)=1.0d0/n
c        x(i)=dlog(x0(i))
        x(i)=-dlog(n+0.0d0)
   14 continue
#ifdef CHECKGRAD 
      call maxentfunc(n,theta_ent,x,xnorm,x0,f,fit,shannon,g,.false.)
      write (iout,*) "checking the gradient"
      do i=1,n
        xi=x(i)
c        x(i)=x(i)*(1.0d0+1.0d-5)
        x(i)=x(i)+1.0d-7
c        qpart=0.0d0
c        do j=1,n
c          qpart=qpart+x(j)
c        enddo
c        x(:n)=x(:n)/qpart
        call maxentfunc(n,theta_ent,x,xnorm,x0,aux,fit,shannon,g_,
     &     .false.)
c        x(i)=xi*(1.0d0-1.0d-5)
c        call maxentfunc(n,theta_ent,x,xnorm,x0,f,fit,shannon,g_,.false.)
c        aux=(aux-f)/(2.0d-5*x(i))
        aux=(aux-f)/1.0d-7
        write (iout,'(i5,f10.2,2f10.3,f10.1)') i,x(i),aux,g(i),
     &    (aux-g(i))/g(i)*100
        x(i)=xi
      enddo
#endif
c     We now write the heading of the output.

c      write (iout,16)
c   16 format(/ 5x, 'Solving maxent problem.')
c      call flush(iout)

c     We start the iteration by initializing task.
 
      task = 'START'

c     ------- The beginning of the loop ----------
  111 continue
      
c     This is the call to the L-BFGS-B code.
 
c      write (iout,*) "Before setulb"
c      call flush(iout)
      call setulb(n,m,x,l,u,nbd,f,g,factr,pgtol,wa,iwa,task,iprint,
     +            csave,lsave,isave,dsave)
c      write (iout,*) "After setulb"
c      call flush(iout)
 
      if (task(1:2) .eq. 'FG') then
 
c        The minimization routine has returned to request the
c        function f and gradient g values at the current x.

c        Compute function value f for the sample problem.
         call maxentfunc(n,theta_ent,x,xnorm,x0,f,fit,shannon,g,.false.)
c        Go back to the minimization routine.
         goto 111

      elseif (task(1:5) .eq. 'NEW_X') then
 
c        The minimization routine has returned with a new iterate,
c        and we have opted to continue the iteration.

         goto 111

      else

c        We terminate execution when task is neither FG nor NEW_X.
c        We print the information contained in the string task
c        if the default output is not used and the execution is
c        not stopped intentionally by the user. 

         if (iprint .le. -1 .and. task(1:4) .ne. 'STOP') write(6,*) task

      endif
c     ---------- The end of the loop -------------
      write (iout,*) "nviol",nviol
      call print_ave_restr_all(n,ib)
#ifdef MPI
c      call ave_restr_all1(-1,n,xnorm,f,g)
#endif
#ifdef DEBUG
      write (iout,*) "Final and original weights"
      do i=1,n
        write (iout,'(i6,2f15.10)') i,xnorm(i),x0(i)
      enddo 
#endif
c Update totfree given the calculated probabilities
      totfreemin=totfreemin+dlog(xnorm(imin))
      do i=1,n
        totfree(i)=totfreemin-dlog(xnorm(i))
      enddo
#ifdef DEBUG
      write (iout,*) "The final totfree array"
      do i=1,n
        write (iout,'(i6,f10.3)') i,totfree(i)
      enddo 
#endif
      return
 
      end
c======================= The end of driver1 ============================

c     --------------------------------------------------------------
c             DESCRIPTION OF THE VARIABLES IN L-BFGS-B
c     --------------------------------------------------------------
c
c     n is an INTEGER variable that must be set by the user to the
c       number of variables.  It is not altered by the routine.
c
c     m is an INTEGER variable that must be set by the user to the
c       number of corrections used in the limited memory matrix.
c       It is not altered by the routine.  Values of m < 3  are
c       not recommended, and large values of m can result in excessive
c       computing time. The range  3 <= m <= 20 is recommended. 
c
c     x is a DOUBLE PRECISION array of length n.  On initial entry
c       it must be set by the user to the values of the initial
c       estimate of the solution vector.  Upon successful exit, it
c       contains the values of the variables at the best point
c       found (usually an approximate solution).
c
c     l is a DOUBLE PRECISION array of length n that must be set by
c       the user to the values of the lower bounds on the variables. If
c       the i-th variable has no lower bound, l(i) need not be defined.
c
c     u is a DOUBLE PRECISION array of length n that must be set by
c       the user to the values of the upper bounds on the variables. If
c       the i-th variable has no upper bound, u(i) need not be defined.
c
c     nbd is an INTEGER array of dimension n that must be set by the
c       user to the type of bounds imposed on the variables:
c       nbd(i)=0 if x(i) is unbounded,
c              1 if x(i) has only a lower bound,
c              2 if x(i) has both lower and upper bounds, 
c              3 if x(i) has only an upper bound.
c
c     f is a DOUBLE PRECISION variable.  If the routine setulb returns
c       with task(1:2)= 'FG', then f must be set by the user to
c       contain the value of the function at the point x.
c
c     g is a DOUBLE PRECISION array of length n.  If the routine setulb
c       returns with taskb(1:2)= 'FG', then g must be set by the user to
c       contain the components of the gradient at the point x.
c
c     factr is a DOUBLE PRECISION variable that must be set by the user.
c       It is a tolerance in the termination test for the algorithm.
c       The iteration will stop when
c
c        (f^k - f^{k+1})/max{|f^k|,|f^{k+1}|,1} <= factr*epsmch
c
c       where epsmch is the machine precision which is automatically
c       generated by the code. Typical values for factr on a computer
c       with 15 digits of accuracy in double precision are:
c       factr=1.d+12 for low accuracy;
c             1.d+7  for moderate accuracy; 
c             1.d+1  for extremely high accuracy.
c       The user can suppress this termination test by setting factr=0.
c
c     pgtol is a double precision variable.
c       On entry pgtol >= 0 is specified by the user.  The iteration
c         will stop when
c
c                 max{|proj g_i | i = 1, ..., n} <= pgtol
c
c         where pg_i is the ith component of the projected gradient.
c       The user can suppress this termination test by setting pgtol=0.
c
c     wa is a DOUBLE PRECISION  array of length 
c       (2mmax + 4)nmax + 11mmax^2 + 8mmax used as workspace.
c       This array must not be altered by the user.
c
c     iwa is an INTEGER  array of length 3nmax used as
c       workspace. This array must not be altered by the user.
c
c     task is a CHARACTER string of length 60.
c       On first entry, it must be set to 'START'.
c       On a return with task(1:2)='FG', the user must evaluate the
c         function f and gradient g at the returned value of x.
c       On a return with task(1:5)='NEW_X', an iteration of the
c         algorithm has concluded, and f and g contain f(x) and g(x)
c         respectively.  The user can decide whether to continue or stop
c         the iteration. 
c       When
c         task(1:4)='CONV', the termination test in L-BFGS-B has been 
c           satisfied;
c         task(1:4)='ABNO', the routine has terminated abnormally
c           without being able to satisfy the termination conditions,
c           x contains the best approximation found,
c           f and g contain f(x) and g(x) respectively;
c         task(1:5)='ERROR', the routine has detected an error in the
c           input parameters;
c       On exit with task = 'CONV', 'ABNO' or 'ERROR', the variable task
c         contains additional information that the user can print.
c       This array should not be altered unless the user wants to
c          stop the run for some reason.  See driver2 or driver3
c          for a detailed explanation on how to stop the run 
c          by assigning task(1:4)='STOP' in the driver.
c
c     iprint is an INTEGER variable that must be set by the user.
c       It controls the frequency and type of output generated:
c        iprint<0    no output is generated;
c        iprint=0    print only one line at the last iteration;
c        0<iprint<99 print also f and |proj g| every iprint iterations;
c        iprint=99   print details of every iteration except n-vectors;
c        iprint=100  print also the changes of active set and final x;
c        iprint>100  print details of every iteration including x and g;
c       When iprint > 0, the file iterate.dat will be created to
c                        summarize the iteration.
c
c     csave  is a CHARACTER working array of length 60.
c
c     lsave is a LOGICAL working array of dimension 4.
c       On exit with task = 'NEW_X', the following information is
c         available:
c       lsave(1) = .true.  the initial x did not satisfy the bounds;
c       lsave(2) = .true.  the problem contains bounds;
c       lsave(3) = .true.  each variable has upper and lower bounds.
c
c     isave is an INTEGER working array of dimension 44.
c       On exit with task = 'NEW_X', it contains information that
c       the user may want to access:
c         isave(30) = the current iteration number;
c         isave(34) = the total number of function and gradient
c                         evaluations;
c         isave(36) = the number of function value or gradient
c                                  evaluations in the current iteration;
c         isave(38) = the number of free variables in the current
c                         iteration;
c         isave(39) = the number of active constraints at the current
c                         iteration;
c
c       See the subroutine setulb.f for a description of other 
c       information contained in isave.
c
c     dsave is a DOUBLE PRECISION working array of dimension 29.
c       On exit with task = 'NEW_X', it contains information that
c       the user may want to access:
c         dsave(2) = the value of f at the previous iteration;
c         dsave(5) = the machine precision epsmch generated by the code;
c         dsave(13) = the infinity norm of the projected gradient;
c
c       See the subroutine setulb.f for a description of other 
c       information contained in dsave.
c
c     --------------------------------------------------------------
c           END OF THE DESCRIPTION OF THE VARIABLES IN L-BFGS-B
c     --------------------------------------------------------------
c
c     << An example of subroutine 'timer' for AIX Version 3.2 >>
c
c     subroutine timer(ttime)
c     double precision ttime
c     integer itemp, integer mclock
c
c     itemp = mclock()
c     ttime = dble(itemp)*1.0d-2
c     return
c     end
c-----------------------------------------------------------------------
      subroutine maxentfunc(n,theta_ent,x,xnorm,x0,f,fit,shannon,g,lprn)
      implicit none
      include 'DIMENSIONS'
      include 'COMMON.IOUNITS'
      integer n
      double precision theta_ent,x(n),xnorm(n),x0(n),f,fit,shannon,g(n)
      double precision qpart
      integer i
      logical lprn
      qpart=0.0d0
      do i=1,n
        xnorm(i)=dexp(x(i))
        qpart=qpart+xnorm(i)
      enddo
      xnorm(:n)=xnorm(:n)/qpart 
c      write (iout,*) "qpart",qpart
c Compute enesemble-averaged restraint violation
      call ave_restr_all1(1,n,xnorm,fit,g)
C Compute the negative of Shannon entropy
      shannon=0.0d0
      do i=1,n
        shannon = shannon + xnorm(i)*dlog(xnorm(i)/x0(i))
      enddo
      if (lprn) write (iout,'(a,f10.5,a,f10.5)') "restraints",fit,
     &     " negative of shannon entropy",shannon
C Add the Shannon entropy to the target function
      f = fit+theta_ent*shannon
c Cdd the gradient of the Shannon entropy to target-function gradient
      do 22 i = 1,n
        g(i)=(g(i)+theta_ent*(dlog(xnorm(i)/x0(i))-shannon))/qpart
        g(i)=dexp(x(i))*g(i)
   22 continue
      return
      end
